unit VisualPointEditor01;

interface
uses SysUtils, Classes, Types, Controls, Graphics, Dialogs, StdCtrls, ExtCtrls;

// ---------------------------------------------------------------------------
//   (TPointEditor)     
// ---------------------------------------------------------------------------
//      
type TPointArray  = array of TPoint;

//      
type TpdCode =(pdNone, pdAdd, pdDel, pdMov);
// pdNone -  
// pdAdd -  " "
// pdDel -  " "
// pdMov -  " "

//     "    "
type TPntEditorNotify = procedure(Sender : TObject; pdCode : TpdCode) of Object;

// ===========================================================================
//    (TQueue2L)      
// ===========================================================================
//   
type
    TpLE = ^TLE;
    TLE   = record
    //    (ver. 3.0)
    fpNext  : TpLE;          //    
    fpPrev  : TpLE;          //    
    fENum   : cardinal;      //   
    fRectLE : TRect;         //   
    fpObj   : pointer;       //   
    fpSelf  : TpLE;          //    ( )
end;

//  ,    
type TQueue2L = class(TObject)
  private
    //     (ver. 3.0)
    fLEKey   : cardinal;  //    (Inc only)
    fLECount : cardinal;  //    
    fpHead   : TpLE;      //   
    fpTail   : TpLE;      //   
    fpCurr   : TpLE;      //     
    //   
    // ( )
    //   (     )
    fpLE     : TpLE;      //   
    // ---------------------------------------------------------
    //  /    
    function  NewLE() : TpLE;
    function  FreeLE (pFreeLE : TpLE) : boolean;

    // ---------------------------------------------------------
    //      (  )
    function  FindLE (pRqLE : TpLE) : TpLE;
    // ---------------------------------------------------------
    //       
    procedure InsLE  (Cmd   : char; pNewLE : TpLE);
    //       
    function  CutLE  (Cmd   : char) : TpLE;
    // ---------------------------------------------------------
    //  PROPETTY
    //      
    function  GetfpObj()  : pointer;
    procedure SetfpObj(RqfpObj : pointer);
    // ---------------------------------------------------------
    //         
    procedure InsNewEL
           ( Cmd     : char;    //     
             pTrgLE  : TpLE;    //     
             pNewLE  : TpLE     //     
           );
    // ---------------------------------------------------------
    //    
    function CutOldLE
           ( Cmd      : char;   //    
             pTrgLE   : TpLE    //     
           ) : TpLE;            //     
  protected
    //        
    procedure FreeQueue();
  public
    // ---------------------------------------------------------
    //  
    procedure Free();
    // ---------------------------------------------------------
    //    
    //  Cmd:
    // 'H' -       (pRef = nil)
    // 'T' -        (pRef = nil)
    // 'P' -       (pRef <> nil)
    // 'N' -      (pRef <> nil)
    // RqRec - Rectangle    
    procedure AddLE(Cmd : char; pRef : pointer; RqRec : TRect);
    // ---------------------------------------------------------
    //    
    //  Cmd:
    // 'H' -      (pRef = nil)
    // 'T' -      (pRef = nil)
    // '' -     (pRef <> nil)
    //
    procedure DelLE(Cmd : char; pRef : pointer);
    // ---------------------------------------------------------
    //    (    )
    //     
    //  Cmd:
    // 'H' -      
    // 'T' -      
    // 'N' -      
    // 'P' -      
    //    (    )
    //
    function Navigate (Cmd : char): pointer;
    // ---------------------------------------------------------
    //        X, Y
    function FindXY (X, Y : integer) : pointer;
    //   C X -    RqX
    function FindXNext (RqX : integer) : pointer;
    //        pRef
    function FindPred (pRef : pointer) : pointer;
    //        pRef
    function FindNext (pRef : pointer) : pointer;
    // ---------------------------------------------------------
    //     
    procedure SetRefRect (pRef : pointer; RefRect : TRect);
    //    
    function GetRefRect (pRef : pointer) : TRect;
    //  X -   
    function GetRefX (pRef : pointer) : integer;
    //  Y -   
    function GetRefY (pRef : pointer) : integer;
    // ---------------------------------------------------------
    //     
    function  GetNumLE (pRef : pointer) : cardinal;
    // ---------------------------------------------------------
    //       Image 
    procedure ShowQueueToImg (RqImage : TImage; RqMark : integer);
    // ---------------------------------------------------------
    //      Y
    procedure InvertYQueue (RqImage : TImage; RqMark : integer);
    // ---------------------------------------------------------
    //    
    property Count : cardinal read fLECount;
    property pHead : TpLE read fpHead;
    property pTail : TpLE read fpTail;
    property pCurr : TpLE read fpCurr;
    // ---------------------------------------------------------
    //    
    property ptObj : pointer read GetfpObj write SetfpObj;

end;

// ===========================================================================
//    (TPointEditor)    
// ===========================================================================
const TxtOfsX = 16;  // -    

type TPointEditor = class (TQueue2L)
  private
    // --------------------
    fEImg    : TImage;   // Image   
    fCCEImg  : TColor;   //   Image  
    fCPoint  : TColor;   //      
    fLMark   : integer;  //     
    fGridNm  : integer;  //   
    fTxtW    : integer;  //    
    fTxtH    : integer;  //    
    // --------------------
    fpdCode  : TpdCode;  //      
    // --------------------
    //       
    fonPointEvent : TPntEditorNotify;
    // --------------------
    fCurrLE  : pointer;  //    
    fCurrX   : integer;  // X -   
    fCurrY   : integer;  // Y -   
    // --------------------
    fPredLE  : pointer;  //    
    fPredX   : integer;  // X -   
    fPredY   : integer;  // Y -   
    // --------------------
    fNextLE  : pointer;  //    
    fNextX   : integer;  // X -   
    fNextY   : integer;  // Y -   
    // --------------------
    fElastic : boolean;  //   
    // --------------------
    fpWRef   : pointer;  //     
    fWRect   : TRect;    //   (   )
    fWBM     : TBitMap;  //  BitMap
    // --------------------
    //   onMouseDown
    procedure MouseDown(Sender: TObject; Button: TMouseButton;
                        Shift: TShiftState; X, Y: Integer);
    //   onMouseMove
    procedure MouseMove(Sender: TObject; Shift: TShiftState;
                        X, Y: Integer);
    //   onMouseUp
    procedure MouseUp(Sender: TObject; Button: TMouseButton;
                      Shift: TShiftState; X, Y: Integer);
    // --------------------
    //      
    procedure RunPointEvent;
    // --------------------
    //  ,        
    procedure SaveImgFragment(X, Y : integer);
    //       
    procedure RestoreImgFragment(X, Y : integer);
    // --------------------
    //   (Head, Tail)   
     procedure PresetStartPoints (Neg : boolean);
     // --------------------
    //   
    procedure ShowGrids();
    //   Image  
    procedure ClearImage();
    // --------------------
    //  Left -   Image    (0..255)
    function LeftToLight (RqLeft : integer) : integer;
    //     (0..255)  Left -  Image
    function LightToLeft (RqLight : integer) : integer;
    //  Top -   Image    (0..255)
    function TopToLight (RqTop : integer) : integer;
    //     (0..255)  Top -  Image
    function LightToTop (RqLight : integer) : integer;
    // --------------------
  public
    // --------------------
    //   TPointEditor
    constructor Greate (RqImage : TImage);
    //   TPointEditor
    procedure Free;
    // --------------------
    //     
    procedure ReShowPointEditor();
    //         
    procedure PointEditorNewDoc(RqNeg : Boolean);
    //  Y       
    procedure InvertDocAndShow();
    //          
    function SavePointArray() : TPointArray;
    //       
    function LoadPointArray(RqArr : TPointArray) : boolean;
    // --------------------
    //       Left, Top ( )
    procedure ViewXYQueue (RqMem : TMemo);
    // --------------------
    //       
    property onPointEvent : TPntEditorNotify read fonPointEvent
                                             write fonPointEvent;
end;

// ===========================================================================
//                 
//                     
// ===========================================================================
//    Image (  )
// Cmd:
// 'L' - - 
// 'R' - - 
// 'G' - - 
// 'B' - - 
//
procedure ImgRowGradient(Cmd : Char; RqImg : TImage);
// ---------------------------------------------------------------------------
//    Image (  )
// Cmd:
// 'L' - - 
// 'R' - - 
// 'G' - - 
// 'B' - - 
//
procedure ImgColGradient(Cmd : Char; RqImg : TImage);


// ===========================================================================
// ===========================================================================
implementation

// ===========================================================================
// ===========================================================================
//   (TQueue2L)  - 
// ===========================================================================
//    
function TQueue2L.NewLE() : TpLE;
begin
   Result := nil;
   try
     New(fpLE);                         //     
     Fillchar(fpLE^, SizeOF(fpLE^),#0);  //   
     fpLE^.fpSelf := fpLE;                //   
     Result     := fpLE;
   except
     ShowMessage ('TQueue2L.NewLE :     !');
   end;
end;
// ---------------------------------------------------------------------------
//   
function TQueue2L.FreeLE (pFreeLE : TpLE) : boolean;
begin
   Result := False;
   if (pFreeLE <> nil)
   then begin
     //    
     if (pFreeLE^.fpSelf = pFreeLE)
     then begin
       try
         Dispose(pFreeLE);      //    
         Result := True;
       except
         ShowMessage ('TQueue2L.FreeLE :    !');
       end;
     end
     else ShowMessage ('TQueue2L.FreeLE :   !');
  end;
end;
// ---------------------------------------------------------------------------
//        
procedure TQueue2L.FreeQueue();
begin
   // ,    
   if (fpHead = nil) and (fpTail = nil) then Exit;
   while fpHead <> nil do
   begin
      fpCurr := fpHead;          //     
      fpHead := fpCurr^.fpNext;  //  fpHead   
      if not FreeLE(fpCurr)      //    
      then begin
         ShowMessage ('TQueue2L.FreeQueue :   '
                    + 'fpCurr  !');
         Exit;
      end
      else Dec(fLECount);        //    
   end;
   //   
   if (fpHead = nil)
   then begin
     fpTail := nil;              //   
     fpCurr := nil;              //   
     fLEKey :=0;                 //    (Inc only)
     fLECount :=0;               //    
   end;
end;
// ---------------------------------------------------------------------------
//  
procedure TQueue2L.Free();
begin
   FreeQueue();
   inherited Free;
end;

// ===========================================================================
//   (TQueue2L)   
// ===========================================================================
//      (  )
function  TQueue2L.FindLE (pRqLE : TpLE) : TpLE;
begin
   Result := nil;
   if fpHead <> nil
   then begin
      fpLE := fpHead;
      try
         repeat
           if fpLE^.fpSelf = pRqLE
           then begin
              //   
              Result:= pRqLE;
              Exit;
           end;
           fpLE := fpLE^.fpNext;
         until (fpLE = nil) or (Result <> nil);
      except {  } end;
   end;
end;
// ---------------------------------------------------------------------------
//       
// Cmd:
//  'H' -      (head)
//  'T' -      (tail)
//
procedure TQueue2L.InsLE
           ( Cmd     : char;
             pNewLE  : TpLE     //     
           );
begin
 // ,    
 if (fpHead = nil) and (fpTail = nil)
 then begin
   // INSERT   
   fpHead := pNewLE;
   fpTail := pNewLE;
 end
 else begin
   // INSERT    
   case UpCase(Cmd) of
   'H': begin
       //      
       fpLE := fpHead;            //     
       fpLE^.fpPrev   := pNewLE;  //      
       pNewLE^.fpNext := fpLE;    //      
       fpHead := pNewLE;         //      
       end;
   'T': begin
       //      
       fpLE := fpTail;            //     
       fpLE^.fpNext := pNewLE;    //      
       pNewLE^.fpPrev := fpLE;    //      
       fpTail := pNewLE;         //      
       end;
   end; // case
 end;
 Inc(fLEKey);                    // +1    
 Inc(fLECount);                  // +1     
 pNewLE^.fENum := fLEKey;         //    
 fpCurr := pNewLE;               //  INSERT   
end;
// ---------------------------------------------------------------------------
//          (Protected)
//  Cmd:
// 'H' -       (pTrgLE = nil)
// 'T' -        (pTrgLE = nil)
// 'P' -       (pTrgLE <> nil)
// 'N' -      (pTrgLE <> nil)
//
procedure TQueue2L.InsNewEL
           ( Cmd     : char;    //     
             pTrgLE  : TpLE;    //     
             pNewLE  : TpLE     //     
           );
begin
 // ,    
 if (fpHead = nil) and (fpTail = nil)
 then begin
   // INSERT   
   fpHead := pNewLE;
   fpTail := pNewLE;
 end
 else begin
   case UpCase(Cmd) of
   'H': begin
         if (pNewLE = nil)           //     
         then Exit;                  //   ( )
         InsLE('H', pNewLE);         //     
         Exit;                       //  
        end;
   'T': begin
         if (pNewLE = nil)           //     
         then Exit;                  //   ( )
         InsLE('T', pNewLE);         //     
         Exit;                       //  
        end;
   'P': begin //     pCurEl - 
          if (pTrgLE = nil) or (pNewLE = nil)
          then Exit;                 //   ( )
          if pTrgLE^.fpPrev = nil
          then begin                 //   
            InsLE('H', pNewLE);      //     
            Exit;                    //  
          end
          else begin
            //      
            pNewLE^.fpNext := pTrgLE;
            pNewLE^.fpPrev := pTrgLE^.fpPrev;
            //      
            fpLE := pTrgLE^.fpPrev;    //    
            fpLE^.fpNext := pNewLE;    //    
            pTrgLE^.fpPrev := pNewLE; //     
          end;
        end;
   'N': begin //     pCurEl - 
          if (pTrgLE = nil) or (pNewLE = nil)
          then Exit;                 //   ( )
          if pTrgLE^.fpNext = nil
          then begin                 //   
             InsLE('T', pNewLE);     //     
             Exit;                   //  
          end
          else begin
            //      
            pNewLE^.fpPrev := pTrgLE;
            pNewLE^.fpNext := pTrgLE^.fpNext;
            //      
            fpLE := pTrgLE^.fpNext;    //    
            fpLE^.fpPrev := pNewLE;    //    
            pTrgLE^.fpNext := pNewLE; //     
          end;
       end;
   end; // case
 end;
 Inc(fLEKey);                         // +1    
 Inc(fLECount);                       // +1     
 pNewLE^.fENum := fLEKey;              //    
 fpCurr := pNewLE;                    //  INSERT   
end;
// ---------------------------------------------------------------------------
//       
//  Cmd:
// 'H' -     
// 'T' -     
//
function TQueue2L.CutLE
     ( Cmd      : char          //  'H' -   head, 'T'  tail 
     ) : TpLE;                  //     
begin
 Result := nil;                 //  
 if (fpHead <> nil) and (fpTail <> nil)
 then begin
   //   
   case UpCase(Cmd) of
   'H': begin  //     
      Result := fpHead;          //      
      fpLE    := fpHead^.fpNext;   //     nil
      if (fpLE = nil)            // Result    ?
      then fpTail := nil
      else fpLE^.fpPrev := nil;
      fpHead := fpLE;             //    
   end;
   'T': begin  //     
      Result := fpTail;          //      
      fpLE    := fpTail^.fpPrev;   //     nil
      if (fpLE = nil)            // Result    ?
      then fpHead := nil
      else fpLE^.fpNext := nil;
      fpTail := fpLE;             //    
   end;
   end; // case
   //     
   fpCurr := fpLE;
   if (fpTail = nil) or (fpHead = nil)
   //   
   then fLECount := 0            //   
   else Dec(fLECount);           //     
 end;
end;
// ---------------------------------------------------------------------------
//    
//  Cmd:
// 'H' -      (pTrgLE = nil)
// 'T' -      (pTrgLE = nil)
// '' -     (pTrgLE <> nil)
//
function TQueue2L.CutOldLE
     ( Cmd      : char;         //    
       pTrgLE   : TpLE          //     
     ) : TpLE;                  //     
begin
 Result := nil;
 if (fpHead <> nil) and (fpTail <> nil)
 then begin
   //   
   case UpCase(Cmd) of
   'H': begin  //     
      Result := CutLE('H');
      Exit;                     //  
   end;
   'T': begin  //     
      Result := CutLE('T');
      Exit;                     //  
   end;
   'C': begin
      //  
      if (pTrgLE = nil) then Exit;
      //    
      if (pTrgLE^.fpPrev = nil) and (pTrgLE = fpHead)
      then begin
         //  pTrgLE    
         Result := CutLE('H');  //     
         Exit;                  //  
      end;
      if (pTrgLE^.fpNext = nil)  and (pTrgLE = fpTail)
      then begin
         //  pTrgLE    
         Result := CutLE('T');  //     
         Exit;                  //  
      end;
      //     
      Result := pTrgLE;
      //    NEXT
      fpLE := pTrgLE^.fpPrev;
      fpLE^.fpNext := pTrgLE^.fpNext;
      //    PREV
      fpLE := pTrgLE^.fpNext;
      fpLE^.fpPrev := pTrgLE^.fpPrev;
   end;
   end; // case
   //       
   if Result = fpCurr then fpCurr := nil;
   //   
   if (fpTail = nil) or (fpHead = nil)
   then fLECount := 0           //   
   else Dec(fLECount);          //     
 end;
end;

// ===========================================================================
//  (TQueue2L)   
// ===========================================================================
//    
//  Cmd:
// 'H' -       (pRef = nil)
// 'T' -        (pRef = nil)
// 'P' -       (pRef <> nil)
// 'N' -      (pRef <> nil)
// RqRec - Rectangle    
procedure TQueue2L.AddLE(Cmd : char; pRef : pointer; RqRec : TRect);
begin
  fpLE := NewLE();     //    
  if fpLE <> nil
  then begin
      fpLE^.fRectLE := RqRec;
      //    
      InsNewEL(Cmd, TpLE(pRef), fpLE)
  end
  else ShowMessage('TQueue2L.AddLE :      ');
end;
// ---------------------------------------------------------------------------
//    
//  Cmd:
// 'H' -      (pRef = nil)
// 'T' -      (pRef = nil)
// '' -     (pRef <> nil)
//
procedure TQueue2L.DelLE(Cmd : char; pRef : pointer);
begin
  case UpCase(Cmd) of
    'H': fpLE := fpHead;
    'T': fpLE := fpTail;
    'C': fpLE := TpLE(pRef);
    else fpLE := nil;
  end;
  if fpLE <> nil
  then begin
     //      
     if fpLE^.fpObj = nil
     then begin
        fpLE := CutOldLE(Cmd, TpLE(pRef));  //    
        if fpLE <> nil then FreeLE(fpLE);   //   
     end
     else ShowMessage('TQueue2L.DelLE :  .' +
                      '  .');
  end;
end;
// ---------------------------------------------------------------------------
//  (TQueue2L)        
// ---------------------------------------------------------------------------
//    (    )
//  Cmd:
// 'H' -      
// 'T' -      
// 'N' -      
// 'P' -      
//
function TQueue2L.Navigate (Cmd : char): pointer;
begin
    Result := nil;
    fpLE := fpCurr;                                  //   
    case UpCase(Cmd) of
    'H': Result := fpHead;                           //  
    'N': if fpLE <> nil then Result := fpLE^.fpNext; //  
    'P': if fpLE <> nil then Result := fpLE^.fpPrev; //  
    'T': Result := fpTail;                           //  
    end;
    //      
    if Result <> nil then fpCurr := TpLE(Result);
end;

// ---------------------------------------------------------------------------
//   (TQueue2L)     
// ---------------------------------------------------------------------------
// ,   X, Y    R
function PointInRect(X, Y : integer; R : TRect): Boolean;
begin
  Result := (X >= R.Left) and (X <= R.Right)
        and (Y >= R.Top)  and (Y <= R.Bottom);
end;
// ---------------------------------------------------------------------------
//        X, Y
function  TQueue2L.FindXY (X, Y : integer) : pointer;
begin
   Result := nil;
   if fpHead <> nil
   then begin
      fpLE := fpHead;
      while (fpLE <> nil)
      do begin
         if PointInRect(X, Y, fpLE^.fRectLE)
         then begin
           Result := fpLE;
           Exit;
         end;
         fpLE := fpLE^.fpNext;
      end;
   end;
end;
// ---------------------------------------------------------------------------
//   C X -    RqX
function  TQueue2L.FindXNext (RqX : integer) : pointer;
begin
   Result := nil;
   if fpHead <> nil
   then begin
      fpLE := fpHead;
      while (fpLE <> nil)
      do begin
         if fpLE^.fRectLE.Left > RqX
         then begin
           Result := fpLE;
           Exit;
         end;
         fpLE := fpLE^.fpNext;
      end;
   end;
end;
// ---------------------------------------------------------------------------
//        pRef
function  TQueue2L.FindPred (pRef : pointer) : pointer;
begin
   Result := nil;
   fpLE := FindLE(TpLE(pRef));
   if fpLE <> nil
   then Result := fpLE^.fpPrev;
end;
// ---------------------------------------------------------------------------
//        pRef
function  TQueue2L.FindNext (pRef : pointer) : pointer;
begin
   Result := nil;
   fpLE := FindLE (TpLE(pRef));
   if fpLE <> nil
   then Result := fpLE^.fpNext;
end;

// ---------------------------------------------------------------------------
//  (TQueue2L)     
// ---------------------------------------------------------------------------
//     
procedure TQueue2L.SetRefRect (pRef : pointer; RefRect : TRect);
begin
  fpLE := FindLE (TpLE(pRef));
  if fpLE <> nil then fpLE^.fRectLE := RefRect;
end;
// ---------------------------------------------------------------------------
//    
function TQueue2L.GetRefRect (pRef : pointer) : TRect;
begin
  Result := Rect(0,0,0,0);
  fpLE := FindLE (TpLE(pRef));
  if fpLE <> nil then Result := fpLE^.fRectLE;
end;
// ---------------------------------------------------------------------------
//  X -   
function TQueue2L.GetRefX (pRef : pointer) : integer;
begin
  Result := 0;
  fpLE := FindLE (TpLE(pRef));
  if fpLE <> nil then Result := fpLE^.fRectLE.Left;
end;
// ---------------------------------------------------------------------------
//  Y -   
function TQueue2L.GetRefY (pRef : pointer) : integer;
begin
  Result := 0;
  fpLE := FindLE (TpLE(pRef));
  if fpLE <> nil then Result := fpLE^.fRectLE.Top;
end;

// ---------------------------------------------------------------------------
//  (TQueue2L)     
// ---------------------------------------------------------------------------
//       
function TQueue2L.GetfpObj()  : pointer;
begin
  Result := nil;
  if fpCurr <> nil then Result := fpCurr^.fpObj;
end;
// ---------------------------------------------------------------------------
//        NIL
procedure TQueue2L.SetfpObj(RqfpObj : pointer);
begin
  if fpCurr <> nil
  then fpCurr^.fpObj := RqfpObj
  else ShowMessage ('TQueue2L.SetfpObj :   !');
end;

// ---------------------------------------------------------------------------
//  (TQueue2L)     
// ---------------------------------------------------------------------------
//     
function  TQueue2L.GetNumLE (pRef : pointer) : cardinal;
begin
  Result := 0;
  fpLE := FindLE (TpLE(pRef));
  if fpLE <> nil then Result := fpLE^.fENum;
end;

// ---------------------------------------------------------------------------
//  (TQueue2L)    
// ---------------------------------------------------------------------------
//       Image
procedure TQueue2L.ShowQueueToImg (RqImage : TImage; RqMark : integer);
begin
   if (Self.fLECount < 2) or (fpHead = nil) then Exit;
   fpLE := fpHead;
   with RqImage.Canvas do
   begin
     MoveTo(fpLE^.fRectLE.Left, fpLE^.fRectLE.Top);
     fpLE := fpLE^.fpNext;
     while fpLE <> nil
     do begin
       LineTo(fpLE^.fRectLE.Left, fpLE^.fRectLE.Top);
       //    ( fpTail)  
       if (RqMark > 0) and (fpLE <> fpTail)
       then Rectangle(fpLE^.fRectLE.Left, fpLE^.fRectLE.Top,
                      fpLE^.fRectLE.Left + RqMark, fpLE^.fRectLE.Top  + RqMark);
       fpLE := fpLE^.fpNext;
     end;
  end;
end;

// ---------------------------------------------------------------------------
//  (TQueue2L)    
// ---------------------------------------------------------------------------
//      
procedure TQueue2L.InvertYQueue (RqImage : TImage; RqMark : integer);
var ImgH : integer;
begin
   if fpHead = nil then Exit;
   ImgH := RqImage.Height;
   fpLE := fpHead;
   while fpLE <> nil
   do begin
      fpLE^.fRectLE.Top    := ImgH - fpLE^.fRectLE.Top;
      fpLE^.fRectLE.Bottom := fpLE^.fRectLE.Top + RqMark;
      fpLE := fpLE^.fpNext;
   end;
end;

// ===========================================================================
//   (TQueue2L) 
// ===========================================================================

// ===========================================================================
//    (TPointEditor)    
// ===========================================================================
//   TPointEditor
constructor TPointEditor.Greate (RqImage : TImage);
begin
   inherited Create();
   //  RqImage   
   fEImg   := RqImage;
   //   BitMap    
   fWBM := nil;
   fWBM := TBitMap.Create;
   fWBM.Transparent := False;
   //   
   fEImg.OnMouseDown := MouseDown;
   fEImg.OnMouseMove := MouseMove;
   fEImg.OnMouseUp   := MouseUp;
   // ------------
   //    
   fpdCode := pdNone;
   fonPointEvent := nil;
   // ------------
   //       
   fTxtW := fEImg.Canvas.TextWidth (' XXX, YYY ');
   fTxtH := fEImg.Canvas.TextHeight('XY') + 2;
   //   BitMap    
   fWBM.Height := fTxtH;
   fWBM.Width  := fTxtW;
   // ------------
   //        
   fGridNm  := 16;
   fLMark := 6;
   //      fEImg
   fCCEImg := clBtnFace;
   fCPoint := clBlack;
   fEImg.Canvas.Pen.Color := fCPoint;
   // ------------
   //          
   PointEditorNewDoc(False);
end;
// ---------------------------------------------------------------------------
//   TPointEditor
procedure TPointEditor.Free;
begin
   if Assigned(fWBM) then fWBM.Free;
   inherited Free;
end;
// ---------------------------------------------------------------------------
//      
procedure TPointEditor.RunPointEvent;
begin
   if Assigned(fonPointEvent)and (fpdCode <> pdNone)
   then fonPointEvent(Self, fpdCode);
end;
// ---------------------------------------------------------------------------
//  (TPointEditor)     
// ---------------------------------------------------------------------------
//   (Head, Tail)   
procedure TPointEditor.PresetStartPoints (Neg : boolean);
begin
   with fEImg
   do begin
      if Neg
      then begin
        //   
        AddLE('H', nil, Rect(0, 0, 0, 0));
        AddLE('T', nil, Rect(Width, Height, Width, Height));
      end
      else begin
        //   
        AddLE('H', nil, Rect(0, Height, 0, Height));
        AddLE('T', nil, Rect(Width, 0, Width, 0));
      end;
   end;
end;
// ---------------------------------------------------------------------------
//   
procedure TPointEditor.ShowGrids();
var StepX, StepY, Ind : integer;
begin
   StepX := fEImg.Width div fGridNm;
   StepY := fEImg.Height div fGridNm;
   fEImg.Canvas.Pen.Color := clGray;
   //   
   for Ind:= 1 to (fGridNm -1)
   do begin
     fEImg.Canvas.MoveTo(Ind * StepX, 0);
     fEImg.Canvas.LineTo(Ind * StepX, fEImg.Height);
     //      
     fEImg.Canvas.TextOut(Ind * StepX + 4, fEImg.Height - 16,
            IntToStr(Round(Ind * 256/fGridNm)));
   end;
   //   
   for Ind:= 1 to (fGridNm -1)
   do begin
     fEImg.Canvas.MoveTo(0, Ind * StepY);
     fEImg.Canvas.LineTo(fEImg.Width, Ind * StepX);
     //      Y
     fEImg.Canvas.TextOut(4, Ind * StepY - 16,
            IntToStr(256 - Round(Ind * 256/fGridNm)));
   end;
   fEImg.Canvas.Pen.Color := fCPoint;
end;
// ---------------------------------------------------------------------------
//   Image      
procedure TPointEditor.ClearImage();
begin
  with fEImg.Canvas do
  begin
    Brush.Style := bsSolid;
    Brush.Color := fCCEImg;
    FillRect(Rect(0,0, fEImg.Width, fEImg.Height));
    ShowGrids();
  end;
end;
// ---------------------------------------------------------------------------
//          
procedure TPointEditor.PointEditorNewDoc(RqNeg : Boolean);
begin
   //      
   FreeQueue();
   //   (Head, Tail)   
   PresetStartPoints (RqNeg);
   //     
   ReShowPointEditor();
end;
// ---------------------------------------------------------------------------
//     
procedure TPointEditor.ReShowPointEditor();
begin
  ClearImage();
  ShowQueueToImg(fEImg, fLMark);
end;

// ---------------------------------------------------------------------------
//  (TPointEditor)   
// ---------------------------------------------------------------------------
//  Left -   Image    (0..255)
function TPointEditor.LeftToLight (RqLeft : integer) : integer;
begin
    Result := 0;
    if fEImg.Width > 0 then Result := Round(RqLeft * 255 / fEImg.Width);
end;
// ---------------------------------------------------------------------------
//     (0..255)  Left -  Image
function TPointEditor.LightToLeft (RqLight : integer) : integer;
begin
    Result := Round(RqLight * fEImg.Width / 255);
end;
// ---------------------------------------------------------------------------
//  Top -   Image    (0..255)
function TPointEditor.TopToLight (RqTop : integer) : integer;
begin
    Result := 0;
    if fEImg.Height > 0 then Result := 255 - Round(RqTop * 255 / fEImg.Height);
end;
// ---------------------------------------------------------------------------
//     (0..255)  Top -  Image
function TPointEditor.LightToTop (RqLight : integer) : integer;
begin
    Result := fEImg.Height - Round(RqLight * fEImg.Height / 255 );
end;
// ---------------------------------------------------------------------------
//  (TPointEditor)    
// ---------------------------------------------------------------------------
//  ,        
procedure TPointEditor.SaveImgFragment(X, Y : integer);
begin
   //    fEImg   BitMap
   fWBM.Canvas.CopyRect(Rect(0, 0, fTxtW, fTxtH),
                        fEImg.Canvas,
                        Rect(X + TxtOfsX, Y, X+ TxtOfsX + fTxtW, Y + fTxtH));
   //   ( X  Y   )
   fEImg.Canvas.TextOut(X + TxtOfsX, Y + 1,
                        IntToStr(LeftToLight(X))
                      + ', '
                      + IntToStr(TopToLight(Y)));
end;
// ---------------------------------------------------------------------------
//       
procedure TPointEditor.RestoreImgFragment(X, Y : integer);
begin
   //    fEImg   BitMap
   fEImg.Canvas.Draw(X + TxtOfsX, Y, fWBM);
end;

// ---------------------------------------------------------------------------
//  (TPointEditor)   
// ---------------------------------------------------------------------------
procedure TPointEditor.MouseDown(Sender: TObject; Button: TMouseButton;
                                 Shift: TShiftState; X, Y: Integer);
begin
  // ------------------
  fpdCode := pdNone;   //     
  // ------------------
  fEImg.Canvas.Pen.Color := fCPoint;
  // ------------------
  fCurrLE := nil;     //    
  fElastic := False;
  // ------------------
  if (Shift = [ssDouble, ssLeft])
  then begin
    //    ()
    fpWRef := FindXNext(X);
    if fpWRef <> nil
    then begin
      fWRect := Rect(X, Y, X + fLMark, Y + fLMark);
      AddLE('P', fpWRef, fWRect);
      fpdCode := pdAdd;   //   
      Exit;
    end;
  end;
  // -------------------------------
  if (Shift = [ssDouble, ssRight])
  then begin
    //    ()
    fpWRef := FindXY(X, Y);
    if fpWRef <> nil then DelLE('C', fpWRef);
    fpdCode := pdDel;   //    
    Exit;
  end;
  // -------------------------------
  if Shift = [ssLeft]
  then begin
     //    ()
     fCurrLE := FindXY(X, Y);
     if (fCurrLE <> nil) and (fCurrLE <> pHead) and (fCurrLE <> pTail)
     then begin
        fEImg.Canvas.Pen.Width := 3;
        fEImg.Canvas.Rectangle(GetRefRect(fCurrLE));
        fEImg.Canvas.Pen.Width := 1;
        //
        fCurrX := GetRefX(fCurrLE);
        fCurrY := GetRefY(fCurrLE);
        // --------------------------------
        //   
        fPredLE := FindPred(fCurrLE);
        if fPredLE <> nil
        then begin
           fPredX := GetRefX(fPredLE);
           fPredY := GetRefY(fPredLE);
        end
        else begin
           fPredX := 0;
           fPredY := 0;
        end;
        // --------------------------------
        fNextLE := FindNext(fCurrLE);
        if fNextLE <> nil
        then begin
           fNextX := GetRefX(fNextLE);
           fNextY := GetRefY(fNextLE);
        end
        else begin
           fNextX := 0;
           fNextY := 0;
        end;
        // --------------------------------
        //   ,   
        SaveImgFragment(fCurrX, fCurrY);
        // ---------------------------------
        //   
        fEImg.Canvas.Pen.Mode := pmNotXor;
        fEImg.Canvas.MoveTo(fPredX, fPredY);
        fEImg.Canvas.LineTo(fCurrX, fCurrY);
        fEImg.Canvas.LineTo(fNextX, fNextY);
        // ---------------------------------
        fElastic := True;   //   
        fpdCode := pdMov;   //  
        // ---------------------------------
     end;
  end;
end;
// ---------------------------------------------------------------------------
//  (TPointEditor)     
// ---------------------------------------------------------------------------
procedure TPointEditor.MouseMove(Sender: TObject; Shift: TShiftState;
                                 X, Y: Integer);
begin
  if fElastic
  then begin
    //    X  
    // X -   . 
    //   
    //   
    if (X > fPredX) and (X < fNextX)
    then begin
      //     Y
      if (Y >= 0) and (Y < fEImg.Height)
      then begin
        //    
        fEImg.Canvas.MoveTo(fPredX, fPredY);
        fEImg.Canvas.LineTo(fCurrX, fCurrY);
        fEImg.Canvas.LineTo(fNextX, fNextY);
        // ---------------------------------
        // /  
        RestoreImgFragment(fCurrX, fCurrY);
        //   ,   
        SaveImgFragment(X, Y);
        // ---------------------------------
        //    
        fEImg.Canvas.MoveTo(fPredX, fPredY);
        fEImg.Canvas.LineTo(X, Y);
        fEImg.Canvas.LineTo(fNextX, fNextY);
        fCurrX := X;
        fCurrY := Y;
      end;
    end;
  end;
end;
// ---------------------------------------------------------------------------
//  (TPointEditor)   
// ---------------------------------------------------------------------------
procedure TPointEditor.MouseUp(Sender: TObject; Button: TMouseButton;
                               Shift: TShiftState; X, Y: Integer);
begin
    fEImg.Canvas.Pen.Mode := pmCopy;
    fElastic := False;
    if fCurrLE <> nil
    then begin
      //     Image
      //   ,   
      //    
      if X < 0 then fCurrX := 0;
      if Y < 0 then fCurrY := 0;
      if X > fEImg.Width  - 2 then fCurrX := fEImg.Width  - 2;
      if Y > fEImg.Height - 2 then fCurrY := fEImg.Height - 2;
      //     
      fWRect := Rect(fCurrX, fCurrY, fCurrX + fLMark, fCurrY + fLMark);
      SetRefRect(fCurrLE, fWRect);
    end;
    //     
    ReShowPointEditor();
    //      
    RunPointEvent;
end;
// ---------------------------------------------------------------------------
//  (TPointEditor)    
// ---------------------------------------------------------------------------
//  Y       
procedure TPointEditor.InvertDocAndShow();
begin
  InvertYQueue(fEImg, fLMark);
  //     
  ReShowPointEditor();
end;

// ---------------------------------------------------------------------------
//  (TPointEditor)       
// ---------------------------------------------------------------------------
//          
function TPointEditor.SavePointArray() : TPointArray;
var Ind : integer;
begin
   SetLength (Result, 0);
   fCurrLE := pHead;
   if fCurrLE <> nil
   then begin
      SetLength (Result, Count);
      Ind := 0;
      while (fCurrLE <> nil) and (Ind <= High(Result))
      do begin
         //    
         fCurrX := GetRefX(fCurrLE);
         fCurrY := GetRefY(fCurrLE);
         //      
         Result[Ind].X := LeftToLight(fCurrX);
         Result[Ind].Y := TopToLight (fCurrY);
         //    
         fCurrLE := FindNext (fCurrLE);
         Inc(Ind);
      end
   end;
end;
// ---------------------------------------------------------------------------
//        
function TPointEditor.LoadPointArray(RqArr : TPointArray) : boolean;
var Ind,
    IMin, IMax : integer;
    Neg        : boolean;
begin
  Result := False;
  // --------  1
  if Length(RqArr) < 2 then Exit;
  // --------   
  IMin := Low(RqArr);
  IMax := High(RqArr);
  if (RqArr[IMin].X = 0) and (RqArr[IMin].Y = 0)
  then Neg := False
  else Neg := True;
  // --------    
  //   
  FreeQueue();
  //   (Head, Tail)   
  PresetStartPoints(Neg);
  //   
  for Ind := (IMin + 1) to (IMax - 1)
  do begin
      //     
      fCurrX := RqArr[Ind].X;
      fCurrY := RqArr[Ind].Y;
      // --------  2
      if (fCurrX >= 0) and (fCurrX <= 255) and
         (fCurrY >= 0) and (fCurrY <= 255)
      then begin
        //      Image
        fCurrX := LightToLeft(fCurrX);
        fCurrY := LightToTop (fCurrY);
        //    ()
        fpWRef := FindXNext(fCurrX);
        if fpWRef <> nil
        then begin
          fWRect := Rect(fCurrX, fCurrY, fCurrX + fLMark, fCurrY + fLMark);
          AddLE('P', fpWRef, fWRect);
        end;
      end
      else ShowMessage ('  X : ' + IntToStr(fCurrX));
  end;
  //     
  ReShowPointEditor();
end;

// ---------------------------------------------------------------------------
//  (TPointEditor)   
// ---------------------------------------------------------------------------
//       Left, Top ( )
procedure TPointEditor.ViewXYQueue (RqMem : TMemo);
begin
   RqMem.Lines.Add('');
   RqMem.Lines.Add('   : ' + IntToStr(Count));
   RqMem.Lines.Add('     Left, Top :');
   if fpHead = nil then Exit;
   fCurrLE := fpHead;
   while fCurrLE <> nil
   do begin
      //    
      fCurrX := GetRefX(fCurrLE);
      fCurrY := GetRefY(fCurrLE);
      //    
      RqMem.Lines.Add( Format(' X = %d', [fCurrX]) + #09
                     + Format(' Y = %d', [fCurrY]) + #09
                     + Format(' Addr = %p', [fCurrLE]));
      //    
      fCurrLE := FindNext (fCurrLE);;
   end;
end;
// ===========================================================================
//    (TPointEditor) 
// ===========================================================================

// ===========================================================================
//                
//                     
// ===========================================================================
// 24.02.2013
//    Image (  )
// Cmd:
// 'L' - - 
// 'R' - - 
// 'G' - - 
// 'B' - - 
//
procedure ImgRowGradient(Cmd : Char; RqImg : TImage);
type TpBArr = ^TBArr;
     TBArr = array[0..32767] of Byte;
var  BMCol, BMRow : integer;
     Scale : double;
     L     : byte;
     WRow, WCol   : integer;
     pRow : TpBArr;
begin
  with RqImg.Picture
  do begin
     BitMap.Height := RqImg.Height;
     BitMap.Width  := RqImg.Width;
     BitMap.PixelFormat := pf24bit;
     BMRow := BitMap.Height;
     BMCol := 3 * BitMap.Width;
     Scale := 256 / BitMap.Height;
     for WRow := 0 to (BMRow - 1)
     do begin
        pRow := BitMap.ScanLine[WRow];
        WCol := 0;
        while WCol < BMCol
        do begin
          L := 255 - Trunc(WRow * Scale);
          case UpCase(Cmd) of
          'L' : begin
                 pRow^[WCol]   := L;
                 pRow^[WCol+1] := L;
                 pRow^[WCol+2] := L;
                end;
          'R' : begin
                 pRow^[WCol]   := 0;
                 pRow^[WCol+1] := 0;
                 pRow^[WCol+2] := L;
                end;
          'G' : begin
                 pRow^[WCol]   := 0;
                 pRow^[WCol+1] := L;
                 pRow^[WCol+2] := 0;
                end;
           'B' : begin
                 pRow^[WCol]   := L;
                 pRow^[WCol+1] := 0;
                 pRow^[WCol+2] := 0;
                end;
          end;
          WCol := WCol + 3;
        end;
     end;
  end;
end;
// ---------------------------------------------------------------------------
// 24.02.2013
//    Image (  )
// Cmd:
// 'L' - - 
// 'R' - - 
// 'G' - - 
// 'B' - - 
//
procedure ImgColGradient(Cmd : Char; RqImg : TImage);
type TpBArr = ^TBArr;
     TBArr = array[0..32767] of Byte;
var  BMCol, BMRow : integer;
     Scale : double;
     L     : byte;
     WRow, WCol   : integer;
     pRow : TpBArr;
begin
  with RqImg.Picture
  do begin
     BitMap.Height := RqImg.Height;
     BitMap.Width  := RqImg.Width;
     BitMap.PixelFormat := pf24bit;
     BMRow := BitMap.Height;
     BMCol := 3 * BitMap.Width;
     Scale := 256 / BitMap.Width;
     for WRow := 0 to (BMRow - 1)
     do begin
        pRow := BitMap.ScanLine[WRow];
        WCol := 0;
        while WCol < BMCol
        do begin
          L := Trunc((WCol div 3) * Scale);
          case UpCase(Cmd) of
          'L' : begin
                 pRow^[WCol]   := L;
                 pRow^[WCol+1] := L;
                 pRow^[WCol+2] := L;
                end;
          'R' : begin
                 pRow^[WCol]   := 0;
                 pRow^[WCol+1] := 0;
                 pRow^[WCol+2] := L;
                end;
          'G' : begin
                 pRow^[WCol]   := 0;
                 pRow^[WCol+1] := L;
                 pRow^[WCol+2] := 0;
                end;
           'B' : begin
                 pRow^[WCol]   := L;
                 pRow^[WCol+1] := 0;
                 pRow^[WCol+2] := 0;
                end;
          end;
          WCol := WCol + 3;
        end;
     end;
  end;
end;

// ===========================================================================
//                     END OF IMPLEMENTATION
// ===========================================================================

end.
